using UnityEditor;
using UnityEditor.Build;
using UnityEngine;
using System.Linq;

namespace NowGG.Sdk.Editor
{
    /// <summary>
    /// Automatically defines UNITY_PURCHASING_V5_OR_NEWER scripting define symbol
    /// based on the installed Unity IAP package version
    /// </summary>
    [InitializeOnLoad]
    public class NowGGIAPVersionDefineSymbol
    {
        private const string DEFINE_V5_SYMBOL = "UNITY_PURCHASING_V5_OR_NEWER";
        private const string DEFINE_V4_SYMBOL = "UNITY_PURCHASING_V4_OR_OLDER";
        private const string IAP_PACKAGE_NAME = "com.unity.purchasing";

        static NowGGIAPVersionDefineSymbol()
        {
            CheckAndUpdateDefineSymbol();
        }

        private static void CheckAndUpdateDefineSymbol()
        {
            bool isVersion5OrNewer = IsUnityIAPVersion5OrNewer();
            UpdateDefineSymbol(isVersion5OrNewer);
        }

        private static bool IsUnityIAPVersion5OrNewer()
        {
            Debug.Log("[NowGG SDK] Checking Unity IAP package version...");

            var packageList = UnityEditor.PackageManager.PackageInfo.GetAllRegisteredPackages();
            
            foreach (var package in packageList)
            {
                if (package.name == IAP_PACKAGE_NAME)
                {
                    string version = package.version;
                    Debug.Log($"[NowGG SDK] Detected Unity IAP version: {version}");
                    
                    // Parse major version
                    if (!string.IsNullOrEmpty(version))
                    {
                        string[] versionParts = version.Split('.');
                        if (versionParts.Length > 0 && int.TryParse(versionParts[0], out int majorVersion))
                        {
                            return majorVersion >= 5;
                        }
                    }
                }
            }
            
            Debug.LogWarning($"[NowGG SDK] Unity IAP package not found. Defaulting to v4 implementation.");
            return false;
        }

        private static void UpdateDefineSymbol(bool isV5OrNewer)
        {
            NamedBuildTarget[] targetGroups = new NamedBuildTarget[]
            {
                NamedBuildTarget.Android,
                NamedBuildTarget.iOS,
                NamedBuildTarget.Standalone
            };

            foreach (var targetGroup in targetGroups)
            {
                string defines = PlayerSettings.GetScriptingDefineSymbols(targetGroup);
                var definesList = defines.Split(';').ToList();
                bool containsV5Define = definesList.Contains(DEFINE_V5_SYMBOL);
                bool containsV4Define = definesList.Contains(DEFINE_V4_SYMBOL);

                if (isV5OrNewer)
                {
                    if (!containsV5Define)
                    {
                        definesList.Add(DEFINE_V5_SYMBOL);
                    }
                    if (containsV4Define)
                    {
                        definesList.Remove(DEFINE_V4_SYMBOL);
                    }
                    string newDefines = string.Join(";", definesList.ToArray());
                    PlayerSettings.SetScriptingDefineSymbols(targetGroup, newDefines);
                    Debug.Log($"[NowGG SDK] Added '{DEFINE_V5_SYMBOL}' and removed '{DEFINE_V4_SYMBOL}' from {targetGroup} scripting define symbols.");
                }
                else
                {
                    if (containsV5Define){
                        definesList.Remove(DEFINE_V5_SYMBOL);
                    }
                    if (!containsV4Define)
                    {
                        definesList.Add(DEFINE_V4_SYMBOL);
                    }
                    string newDefines = string.Join(";", definesList.ToArray());
                    PlayerSettings.SetScriptingDefineSymbols(targetGroup, newDefines);
                    Debug.Log($"[NowGG SDK] Removed '{DEFINE_V5_SYMBOL}'  and added '{DEFINE_V4_SYMBOL}' to {targetGroup} scripting define symbols.");
                }
            }

            if (isV5OrNewer)
            {
                Debug.Log("[NowGG SDK] Unity IAP 5.x or newer detected. NowGGStorev5 will be used.");
            }
            else
            {
                Debug.Log("[NowGG SDK] Unity IAP 4.x detected. NowGGStore (v4) will be used.");
            }
        }

        [MenuItem("NowGG SDK/Refresh IAP Version Detection")]
        private static void ManualRefresh()
        {
            Debug.Log("[NowGG SDK] Manually refreshing IAP version detection...");
            CheckAndUpdateDefineSymbol();
        }
    }
}
