#if UNITY_PURCHASING_V5_OR_NEWER
using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Purchasing;
using UnityEngine.Purchasing.Extension;

namespace NowGG.Sdk
{
    public class NowGGStoreV5 : Store
    {
        public static readonly string StoreName = "NowGGStore";
        public ConnectionState State { get; private set; } = ConnectionState.Disconnected;
        public string PaymentId { get; private set; }

        private bool isInitialized = false;

        private ICart cart;
        private string lastUsedProductId = string.Empty;
        private StoreController _storeController;


        // This is where you'd hold references to your SDK / HTTP client, etc.
        // private readonly IMyBackendClient _backend;

        public NowGGStoreV5(string paymentId)
        {
            NowGGLogger.Log("NowGGStoreV5 constructor called");
            PaymentId = paymentId;
            NowGGPaymentsSdkManager.Instance.OnInitSuccess += OnNowGGSdkInitSuccess;
            NowGGPaymentsSdkManager.Instance.OnInitFailed += OnNowGGSdkInitFailed;
            NowGGPaymentsSdkManager.Instance.OnPurchaseCompleted += OnNowGGSdkPurchaseProduct;
            NowGGPaymentsSdkManager.Instance.OnPurchaseFailed += OnNowGGSdkPurchaseFailed;
            NowGGPaymentsSdkManager.Instance.OnConfirmPendingPurchase += OnNowGGSdkConfirmPendingPurchase;
            
        }

        private void EnsureStoreController()
        {
            if (_storeController == null)
            {
                // If you've called SetStoreAsDefault("NOWGG_STORE"), you can
                // use StoreController() without args instead.
                _storeController = UnityIAPServices.StoreController(StoreName);
            }
        }

        // Called by Unity IAP when it wants you to connect to *your* store
        public override void Connect()
        {
            NowGGLogger.Log("NowGGStoreV5 Connect called");
            State = ConnectionState.Connecting;
            NowGGPaymentsSdkManager.Instance.InitializeIap(PaymentId);
        }

        // Unity IAP asks for product metadata (prices, titles, etc.)
        public override void FetchProducts(IReadOnlyCollection<ProductDefinition> products)
        {
            NowGGLogger.Log("NowGGStoreV5 FetchProducts called $products.Count: " + products.Count);
            Dictionary<string, ProductType> newProducts = new Dictionary<string, ProductType>();
            for (int i = 0; i < products.Count; i++)
            {
                var product = products.ElementAt(i);

                if (product.type != UnityEngine.Purchasing.ProductType.Consumable)
                {
                    var error = $"Nowgg currently doesn't support product types other than consumable. Purchasing will not work for product: {product.id}.";
                    NowGGLogger.LogError(error);
                }

                newProducts.Add(product.id, NowGG.Sdk.ProductType.Consumable); 

            }

             NowGGPaymentsSdkManager.Instance.FetchAdditionalProducts(newProducts,
                () =>
                {
                    NowGGLogger.Log("Fetching additional products succeeded.");
                    //Additional products fetched can now be purchased.
                    List<ProductDescription> productsFetched = new List<ProductDescription>();
                    if (NowGGPaymentsSdkManager.Instance.products.Length > 0)
                    {
                        NowGGLogger.Log($"Number of products fetched: {NowGGPaymentsSdkManager.Instance.products.Length}");
                        for (int i = 0; i < NowGGPaymentsSdkManager.Instance.products.Length; i++)
                        {
                            var initialProduct = NowGGPaymentsSdkManager.Instance.products[i];
                            var productMetadata = new ProductMetadata(initialProduct.price, 
                                initialProduct.title, initialProduct.description, 
                                initialProduct.priceCurrencyCode, 
                                decimal.Parse(initialProduct.priceAmountMicros) / 1000000);   
                            ProductDescription finalProduct = new(initialProduct.productId, productMetadata);
                            productsFetched.Add(finalProduct);
                        }
                    }
                    ProductsCallback?.OnProductsFetched(productsFetched);       
                },
                reason =>
                {
                    NowGGLogger.Log($"Fetching additional products failed: {reason}");
                    ProductsCallback?.OnProductsFetchFailed(new ProductFetchFailureDescription(ProductFetchFailureReason.ProductsUnavailable, reason));
                });
        }

        // Unity IAP asks you to fetch existing purchases (restore / entitlement sync)
        public override void FetchPurchases()
        {
            // var purchases = await _backend.FetchPurchasesAsync();

            // Convert your response to a list of ConfirmedOrder (and maybe PendingOrder)
            // PurchaseFetchCallback?.OnPurchasesRetrieved(orders);
            NowGGLogger.Log("NowGGStoreV5 FetchPurchases called");
            NowGGPaymentsSdkManager.Instance.QueryPendingPurchases();
            // On error:
            // PurchaseFetchCallback?.OnPurchasesFailed(
            //     new PurchasesFetchFailureDescription(PurchasesFetchFailureReason.Unknown, "message"));
        }

        // Unity IAP wants you to start a purchase
        public override void Purchase(ICart cart)
        {   
            NowGGLogger.Log("NowGGStoreV5 Purchase called");
            this.cart = cart;
            cart.Items().ToList().ForEach(item =>
            {
                lastUsedProductId = item.Product.definition.storeSpecificId;
                NowGGLogger.Log($"Starting purchase for product: {item.Product.definition.storeSpecificId}");
                NowGGPaymentsSdkManager.Instance.PurchaseProduct(item.Product.definition.storeSpecificId, null);
            });
           
        }

        // Unity IAP is confirming (finishing) a pending transaction
        public override void FinishTransaction(PendingOrder pendingOrder)
        {
            NowGGLogger.Log($"Finishing transaction for order with TransactionID: {pendingOrder.Info.TransactionID}");
            NowGGPaymentsSdkManager.Instance.ConfirmPendingPurchase(pendingOrder.Info.TransactionID, ProductType.Consumable);
        }

        // Optional: entitlement check for a single product (e.g. subscriptions)
        public override void CheckEntitlement(ProductDefinition product)
        {
            // _backend.CheckEntitlement(product.id, (entitled, extra) => { ... });

            // EntitlementCallback?.OnCheckEntitlementSucceeded(
            //     new EntitlementCheckResult(product, entitled, extraData));
            // Or EntitlementCallback?.OnCheckEntitlementFailed(...);
        }


        private void OnNowGGSdkInitSuccess()
        {   
            NowGGLogger.Log("NowGGStoreV5 OnNowGGSdkInitSuccess called");
            if (!isInitialized){
                State = ConnectionState.Connected;
                ConnectCallback?.OnStoreConnectionSucceeded();
                isInitialized = true;
            }
            
            NowGGLogger.Log("NowggSdk initialization success");
            List<ProductDescription> productsFetched = new List<ProductDescription>();
            if (NowGGPaymentsSdkManager.Instance.products.Length > 0)
            {
                for (int i = 0; i < NowGGPaymentsSdkManager.Instance.products.Length; i++)
                {
                    var initialProduct = NowGGPaymentsSdkManager.Instance.products[i];
                    var productMetadata = new ProductMetadata(initialProduct.price,
                        initialProduct.title, initialProduct.description,
                        initialProduct.priceCurrencyCode,
                        decimal.Parse(initialProduct.priceAmountMicros) / 1000000);

                    ProductDescription finalProduct = new(initialProduct.productId, productMetadata);
                    productsFetched.Add(finalProduct);
                }
            }
            NowGGLogger.Log($"NowGGStoreV5 fetched {productsFetched.Count} products on init success.");
            ProductsCallback?.OnProductsFetched(productsFetched);
            
        }

        private void OnNowGGSdkInitFailed(string error)
        {
            NowGGLogger.Log("NowGGStoreV5 OnNowGGSdkInitFailed called");
            isInitialized = false;
            State = ConnectionState.Disconnected;
            ConnectCallback?.OnStoreConnectionFailed(new StoreConnectionFailureDescription(error));
            NowGGLogger.Log($"NowggSdk initialization failed with error: {error}");
        }

        PurchaseProcessingResult OnNowGGSdkPurchaseProduct(PurchasedProduct purchasedProduct)
        {
            NowGGLogger.Log("NowGGStoreV5 OnNowGGSdkPurchaseProduct called for product: " + purchasedProduct.productId);
            try
            {
                NowGGLogger.Log($"Purchased product: {purchasedProduct.productId}");

                //acknowledgedTransactions.Add(purchasedProduct.purchaseToken, purchasedProduct.isAcknowledged);

                Dictionary<string, object> receiptDict = new Dictionary<string, object>();

                // Required for purchase validation
                receiptDict.Add("productId", purchasedProduct.productId);
                receiptDict.Add("purchaseToken", purchasedProduct.purchaseToken);
                receiptDict.Add("json", purchasedProduct.originalJson);
                receiptDict.Add("signature", purchasedProduct.signature);
                string receipt = MiniJson.JsonEncode(receiptDict);
                NowGGLogger.Log($"Receipt generated for product: {purchasedProduct.productId}");
                var purchasInfo = new NowGGPurchasedProductInfo(purchasedProduct.productId, true);
                NowGGLogger.Log($"Created NowGGPurchasedProductInfo for product: {purchasedProduct.productId}");
                var orderInfo = new NowGGOrderInfo(receipt, purchasedProduct.purchaseToken,  new List<IPurchasedProductInfo> { purchasInfo }); 
                NowGGLogger.Log($"Created NowGGOrderInfo for product: {purchasedProduct.productId}");
                if (cart == null)
                {  
                    EnsureStoreController();
                    var product = _storeController.GetProductById(purchasedProduct.productId);
                    cart = new NowGGCart(new List<UnityEngine.Purchasing.Product> { product });
                }
                var pendingOrder = new PendingOrder(cart, orderInfo);
                NowGGLogger.Log($"Created PendingOrder for product: {purchasedProduct.productId}");
                PurchaseCallback?.OnPurchaseSucceeded(pendingOrder);
                NowGGLogger.Log($"Purchase succeeded for product: {purchasedProduct.productId}");
                cart = null;
            }
            catch (Exception e)
            {
                NowGGLogger.Log($"Error in IStore module: {e.ToString()}");
            }

            return PurchaseProcessingResult.Pending;
        }

        
        private void OnNowGGSdkPurchaseFailed(int errorCode, string errorMessage)
        {
            NowGGLogger.Log("NowGGStoreV5 OnNowGGSdkPurchaseFailed called for product: " + lastUsedProductId);
            PurchaseFailureReason failureReason = MapErrorCodeToFailureReason(errorCode);

            NowGGLogger.Log($"Purchase failed for product: {lastUsedProductId} Reason: {errorMessage}");
            PurchaseCallback?.OnPurchaseFailed(new FailedOrder(cart,failureReason, errorMessage));
        }

        private void OnNowGGSdkConfirmPendingPurchase(string token, int result)
        {
            NowGGLogger.Log($"OnNowGGSdkConfirmPendingPurchase called with token: {token}, result: {result}");
            if (result == 0)
            {
                ConfirmCallback?.OnConfirmOrderSucceeded(token);
            }
            else
            {
                ConfirmCallback?.OnConfirmOrderFailed(new FailedOrder(cart, MapErrorCodeToFailureReason(result), $"Confirm pending purchase failed with result code: {result}"));
            }
        }

        private PurchaseFailureReason MapErrorCodeToFailureReason(int errorCode)
        {
            if (errorCode == 1 || errorCode ==109)
            {
                return PurchaseFailureReason.UserCancelled;
            }
            else if (errorCode == 2 || errorCode == 3)
            {
                return PurchaseFailureReason.PurchasingUnavailable;
            }
            else if (errorCode == 4 || errorCode == 101)
            {
                return PurchaseFailureReason.ProductUnavailable;
            }
            else if (errorCode == 9)
            {
                return PurchaseFailureReason.PaymentDeclined;
            }else if (errorCode == 102)
            {
                return PurchaseFailureReason.ValidationFailure;
            }
            else if (errorCode == 103)
            {
                return PurchaseFailureReason.DuplicateTransaction;
            }
            else if (errorCode == 104)
            {
                return PurchaseFailureReason.DuplicateTransaction;
            }
            else if (errorCode == 105)
            {
                return PurchaseFailureReason.ExistingPurchasePending;
            }
            else
            {
                return PurchaseFailureReason.Unknown;
            }
        }



    }

  


}
#endif
