# NowGG Store - Unity IAP Version Support

This folder contains implementations of the NowGG Custom Store for different versions of Unity IAP.

## Supported Versions

### Unity IAP 4.x (versions < 5.0.0)
- **Files**: `NowGGStore.cs`, `NowGGPurchasingModule.cs`
- **Location**: `CustomStore/` (root level)
- **API**: Uses the older `OnPurchaseSucceeded(string productId, string receipt, string transactionId)` signature

### Unity IAP 5.x (versions >= 5.0.0)
- **Files**: `NowGGStorev5.cs`, `NowGGPurchasingModulev5.cs`
- **Location**: `CustomStore/NowGGStorev5/`
- **API**: Uses the newer `OnPurchaseSucceeded(PurchaseReceipt receipt)` signature

## How It Works

The NowGG SDK uses **conditional compilation** to automatically select the correct store implementation based on your Unity IAP version:

1. An Editor script (`NowGGIAPVersionDefineSymbol.cs`) automatically detects your Unity IAP package version
2. If Unity IAP 5.x or newer is detected, it adds the `UNITY_PURCHASING_V5_OR_NEWER` scripting define symbol
3. The appropriate store implementation is compiled based on this symbol
4. Only one version is active at a time - no manual selection needed!

## Usage

Simply use the `NowGGStoreVersionDetector` helper to get the correct module:

```csharp
using NowGG.Sdk;
using UnityEngine.Purchasing;

public class YourIAPManager : IStoreListener
{
    void InitializePurchasing()
    {
        var builder = ConfigurationBuilder.Instance(
            NowGGStoreVersionDetector.GetNowGGPurchasingModule()
        );
        
        // Add your products
        builder.AddProduct("product_id", ProductType.Consumable);
        
        // Initialize Unity IAP
        UnityPurchasing.Initialize(this, builder);
    }
    
    // Implement IStoreListener methods...
}
```

That's it! The SDK automatically handles version detection and uses the correct implementation.

## Key Differences Between Versions

### Unity IAP 4.x
- `IStoreCallback.OnPurchaseSucceeded(string productId, string receipt, string transactionId)`
- Receipt data passed as separate parameters

### Unity IAP 5.x
- `IStoreCallback.OnPurchaseSucceeded(PurchaseReceipt receipt)`
- Receipt data encapsulated in a `PurchaseReceipt` object
- More structured and type-safe API

## Automatic Version Detection

The SDK automatically detects your Unity IAP version through:
1. **Editor Script**: `NowGGIAPVersionDefineSymbol.cs` runs when Unity loads
2. **Package Detection**: Reads the installed Unity IAP package version
3. **Define Symbol**: Sets `UNITY_PURCHASING_V5_OR_NEWER` if version 5.x+ is detected
4. **Conditional Compilation**: Compiles only the appropriate store implementation

### Manual Refresh

If you update your Unity IAP package and need to refresh version detection:
- Go to menu: **NowGG SDK → Refresh IAP Version Detection**
- Or reimport the NowGG SDK

## Integration Benefits

✅ **Zero Configuration**: Automatic version detection  
✅ **Single SDK Package**: Both v4 and v5 implementations included  
✅ **No Breaking Changes**: Seamlessly supports upgrades from Unity IAP 4.x to 5.x  
✅ **Clean API**: Use `NowGGStoreVersionDetector.GetNowGGPurchasingModule()` everywhere  
✅ **Future-Proof**: Easy to extend for future Unity IAP versions

## Troubleshooting

### Compilation Errors After Updating Unity IAP

1. Check your Unity IAP package version in Package Manager
2. Go to **NowGG SDK → Refresh IAP Version Detection** from the menu
3. Wait for Unity to recompile
4. If issues persist, clear `Library/ScriptAssemblies` and reimport

### Verifying Active Version

Check the Unity Console for messages like:
```
[NowGG SDK] Detected Unity IAP version: 5.0.3
[NowGG SDK] Unity IAP 5.x or newer detected. NowGGStorev5 will be used.
```

### Common Issues

**Problem**: Getting "Type not found" errors  
**Solution**: The wrong version is compiled. Run **NowGG SDK → Refresh IAP Version Detection**

**Problem**: Both versions trying to compile  
**Solution**: This shouldn't happen with conditional compilation. Check for duplicate define symbols in Player Settings

## Technical Details

### Conditional Compilation Directives

- v4 files: `#if !UNITY_PURCHASING_V5_OR_NEWER`
- v5 files: `#if UNITY_PURCHASING_V5_OR_NEWER`

### Scripting Define Symbol

- Symbol: `UNITY_PURCHASING_V5_OR_NEWER`
- Applied to: Android, iOS, Standalone build targets
- Managed by: `NowGGIAPVersionDefineSymbol.cs` Editor script

For support, please contact the NowGG SDK team.
