﻿using System;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Purchasing;

namespace NowGG.Sdk
{

    public interface NowGGPaymentsListener
    {
        void OnNowGGInitSuccess(string response);
        void OnNowGGInitFailed(string error);
        void OnNowGGPurchaseUpdated(string details);
        void OnNowGGPurchaseFailed(int errorCode, string errorMessage);
        void OnNowGGConsumeFinished(string token, int result);

        void OnNowGGAcknowledgeFinished(int result);

        void OnNowGGFetchingAdditionalProductsSuccess(string response);
        void OnNowGGFetchingAdditionalProductsFailed(string error);
    }


    public class NowGGPaymentsSdkManager : AndroidJavaProxy, NowGGPaymentsListener
    {
        AndroidJavaObject nowGGAndroidJavaObj;
        AndroidJavaObject nowGGDetectionJavaObj;

        public Action OnInitSuccess { get; set; }
        public Action<string> OnInitFailed { get; set; }
        public string PaymentId { get; set; }
        public string InGameId { get; set; }

        public delegate PurchaseProcessingResult OnPurchaseCompletedDelegate(PurchasedProduct purchase);
        public OnPurchaseCompletedDelegate OnPurchaseCompleted;
        public Action<int, string> OnPurchaseFailed;

        public Action<string, int> OnConfirmPendingPurchase;

        private Action OnFetchingAdditionalProductsSuccess;
        private Action<string> OnFetchingAdditionalProductsFailed;



        public Product[] products;

        public bool IsInitialized { get; set; }
        public bool IsGuest { get; set; }


        private static NowGGPaymentsSdkManager instance;
        public static NowGGPaymentsSdkManager Instance
        {
            get
            {
                if (instance == null)
                {
                    instance = new NowGGPaymentsSdkManager();
                }
                return instance;
            }
        }

        public static bool IsNowGGIapAvailable()
        {
            bool isNowGGIapAvailable = true;
#if !UNITY_EDITOR
            AndroidJavaClass player = new  AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject activity = player.GetStatic<AndroidJavaObject>("currentActivity");

            AndroidJavaClass nowGGClass = new  AndroidJavaClass("gg.now.sdk.NowggPaymentUnitySDK");
            isNowGGIapAvailable = nowGGClass.CallStatic<bool>("checkIAPAvailability",activity);
#endif
            NowGGLogger.Log("is NowGG IAP available.." + isNowGGIapAvailable);
            return isNowGGIapAvailable;
        }

        public NowGGPaymentsSdkManager() : base("gg.now.sdk.common.UnityPaymentsCallBack")
        {
        }

        public void InitializeIap(string appId, string inGameId)
        {
            this.products = new Product[NowGGProductBuilder.Instance.GetProductCount()];
#if UNITY_PURCHASING_V4_OR_OLDER
            if (NowGGProductBuilder.Instance.GetProductCount() == 0)
            {
                OnInitFailed("Product id list is empty");
                NowGGLogger.Log("Product id list is empty");
                return;
            }
#endif
            nowGGAndroidJavaObj = new AndroidJavaObject("gg.now.sdk.NowggPaymentUnitySDK");

#if !UNITY_EDITOR
            AndroidJavaClass player = new  AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject activity = player.GetStatic<AndroidJavaObject>("currentActivity");
            NowGGLogger.Log(appId);
            NowGGLogger.Log(inGameId);
            if(inGameId == null || inGameId.Length == 0 ){
              nowGGAndroidJavaObj.Call("initialize", activity, this, JsonHelper.ToJson(NowGGProductBuilder.Instance.GetProductIdArray()), appId, null);
              IsGuest = false;
            }else{
              nowGGAndroidJavaObj.Call("initialize", activity, this, JsonHelper.ToJson(NowGGProductBuilder.Instance.GetProductIdArray()), appId, inGameId);
              IsGuest = true;
            }
#else
            /// make demo product json string for editor testing
            Product[] products = new Product[NowGGProductBuilder.Instance.GetProductCount()];
            var productIdArray = NowGGProductBuilder.Instance.GetProductIdArray();
            for (int i = 0; i < NowGGProductBuilder.Instance.GetProductCount(); i++)
            {
                products[i] = new Product(productIdArray[i]);
            }
            OnNowGGInitSuccess(JsonHelper.ToJson<Product>(products));
#endif
        }
        public void InitializeIap(string appId)
        {
            this.InitializeIap(appId, null);
        }

        public void OnNowGGInitFailed(string error)
        {
            NowGGLogger.Log($"OnNowGGInitFailed called with error: {error}");
            IsInitialized = false;
            if (OnInitFailed != null)
            {
                OnInitFailed(error);
                NowGGLogger.Log("OnInitFailed callback invoked with error: " + error);
            }
            else
            {
                NowGGLogger.Log("OnInitFailed callback is null.");
            }
        }

        public void OnNowGGInitSuccess(string response)
        {
            NowGGLogger.Log($"OnNowGGInitSuccess called with response: {response}");
            IsInitialized = true;
#if !UNITY_EDITOR
            response = "{\"Items\":" + response + "}";
#endif
            products = JsonHelper.FromJson<Product>(response);
            if (OnInitSuccess != null)
            {
                OnInitSuccess();
                NowGGLogger.Log("OnInitSuccess callback invoked.");
            }
            else
            {
                NowGGLogger.Log("OnInitSuccess callback is null.");
            }
        }

        public void PurchaseProduct(string productId, string developerPayload)
        {
            if (IsInitialized == false)
            {
                NowGGLogger.Log("PurchaseProduct FAIL. Not initialized.");
                return;
            }

            bool isValidProduct = false;
            foreach (Product product in products)
            {
                if (string.Equals(productId, product.productId))
                {
                    isValidProduct = true;
                    break;
                }
            }

            if (isValidProduct == false)
            {
                NowGGLogger.Log("Product is not available for purchase");
                return;
            }

            string type = GetProductWithID(productId)?.type;
            NowGGLogger.Log("Purchasing product id " + productId);

#if !UNITY_EDITOR
            nowGGAndroidJavaObj.Call("purchase", productId, type, developerPayload);
#else
            PurchasedProduct[] purchases = { new PurchasedProduct(productId) };
            OnNowGGPurchaseUpdated(JsonHelper.ToJson<PurchasedProduct>(purchases));
#endif
        }

        private void ConsumeProduct(string token)
        {
#if !UNITY_EDITOR
            nowGGAndroidJavaObj.Call("consume", token);
#else
            OnNowGGConsumeFinished("EditorTesting", 0);
#endif
        }

        private void AcknowledgePurchase(string token)
        {
#if !UNITY_EDITOR
            nowGGAndroidJavaObj.Call("acknowledge", token);
#else
            OnNowGGAcknowledgeFinished(0);
#endif
        }

        public void OnNowGGPurchaseUpdated(string details)
        {
            NowGGLogger.Log("OnNowGGProductPurchased: " + details);
#if !UNITY_EDITOR
            details = "{\"Items\":" + details + "}";
#endif
            try
            {
                PurchasedProduct[] purchases = JsonHelper.FromJson<PurchasedProduct>(details);

                foreach (PurchasedProduct purchase in purchases)
                {
                    try
                    {
                        Debug.Log("OnNowGGPurchaseUpdated : " + purchase.productId);
                        Debug.Log("OnNowGGPurchaseUpdated : " + purchase.skuDetails);
                        PurchaseProcessingResult result = OnPurchaseCompleted(purchase);
                        if (result == PurchaseProcessingResult.Complete
                            && (NowGGProductBuilder.Instance.GetProductType(purchase.productId) == ProductType.Consumable))
                        {
                            NowGGLogger.Log($"Consuming product: {purchase.productId}");
                            try
                            {
                                //ServerVerification(purchase.purchaseToken, purchase.productId, "KRW");
                                ConsumeProduct(purchase.purchaseToken);
                            }
                            catch (Exception e)
                            {
                                NowGGLogger.Log("Error consuming purchase: " + e.Message);
                            }
                        }
                        else if (result == PurchaseProcessingResult.Complete && !purchase.isAcknowledged)
                        {
                            try
                            {
                                AcknowledgePurchase(purchase.purchaseToken);
                            }
                            catch (Exception e)
                            {
                                NowGGLogger.Log("Error acknowledging purchase: " + purchase.purchaseToken + " " + e.Message);
                            }

                        }
                    }
                    catch (Exception e)
                    {
                        NowGGLogger.Log("Error processing purchase: " + e.Message);
                    }

                }
            }
            catch (Exception e)
            {
                NowGGLogger.Log("Error parsing purchase details: " + e.Message);
            }

        }

        public void OnNowGGPurchaseFailed(int errorCode, string errorMessage)
        {
            NowGGLogger.Log($"OnNowGGPurchaseFailed called with errorCode: {errorCode}, errorMessage: {errorMessage}");
            if (OnPurchaseFailed != null)
            {
                OnPurchaseFailed(errorCode, errorMessage);
                NowGGLogger.Log("OnPurchaseFailed callback invoked.");
            }
            else
            {
                NowGGLogger.Log("OnPurchaseFailed callback is null.");
            }
        }

        public void OnNowGGConsumeFinished(string token, int result)
        {
            NowGGLogger.Log($"OnNowGGConsumeFinished called with token: {token}, result: {result}");
            if (OnConfirmPendingPurchase != null)
            {
                OnConfirmPendingPurchase(token, result);
                NowGGLogger.Log("OnConfirmPendingPurchase callback invoked.");
            }
            else
            {
                NowGGLogger.Log("OnConfirmPendingPurchase callback is null.");
            }
        }

        public void OnNowGGAcknowledgeFinished(int result)
        {
            NowGGLogger.Log($"OnNowGGAcknowledgeFinished called with result: {result}");
        }

        public void OnNowGGFetchingAdditionalProductsSuccess(string response)
        {
            NowGGLogger.Log($"OnNowGGFetchingAdditionalProductsSuccess called with response: {response}");
            IsInitialized = true;
#if !UNITY_EDITOR
            response = "{\"Items\":" + response + "}";
#endif
            products = JsonHelper.FromJson<Product>(response);

            if (OnFetchingAdditionalProductsSuccess != null)
            {
                OnFetchingAdditionalProductsSuccess();
                NowGGLogger.Log("OnFetchingAdditionalProductsSuccess callback invoked.");
            }
            else
            {
                NowGGLogger.Log("OnFetchingAdditionalProductsSuccess callback is null.");
            }
        }

        public void OnNowGGFetchingAdditionalProductsFailed(string error)
        {
            NowGGLogger.Log($"OnNowGGFetchingAdditionalProductsFailed called with error: {error}");
            if (OnFetchingAdditionalProductsFailed != null)
            {
                OnFetchingAdditionalProductsFailed(error);
                NowGGLogger.Log("OnFetchingAdditionalProductsFailed callback invoked.");
            }
            else
            {
                NowGGLogger.Log("OnFetchingAdditionalProductsFailed callback is null.");
            }
        }

        public void ConfirmPendingPurchase(string purchasetoken, ProductType productType)
        {
            if (productType == ProductType.Consumable)
            {
                ConsumeProduct(purchasetoken);
            }
            else
            {
                AcknowledgePurchase(purchasetoken);
            }
        }

        public void ConfirmPendingPurchase(PurchasedProduct purchasedProduct)
        {
            if (NowGGProductBuilder.Instance.GetProductType(purchasedProduct.productId) == ProductType.Consumable)
            {
                ConsumeProduct(purchasedProduct.purchaseToken);
            }
            else
            {
                AcknowledgePurchase(purchasedProduct.purchaseToken);
            }

        }

        public void QueryPendingPurchases()
        {
            nowGGAndroidJavaObj.Call("queryPendingPurchases");
        }

        public void FetchAdditionalProducts(Dictionary<string, ProductType> additionalProductDict, Action successCallback, Action<string> failCallback)
        {
            OnFetchingAdditionalProductsSuccess = successCallback;
            OnFetchingAdditionalProductsFailed = failCallback;
            NowGGProductBuilder.Instance.AddAdditionalProducts(additionalProductDict);
            nowGGAndroidJavaObj.Call("fetchAdditionalProducts", JsonHelper.ToJson(NowGGProductBuilder.Instance.GetProductIdArray()));
        }

        public Product GetProductWithID(string productId)
        {
            foreach (Product product in products)
            {
                if (string.Equals(productId, product.productId))
                {
                    return product;
                }
            }
            return null;
        }

        public string GetServerVerificationJSON(PurchaseEventArgs args)
        {
            return "{\"purchaseToken\" : \"" + args.purchasedProduct.transactionID + "\"}";
        }

        public static bool IsNowGGPlatform()
        {
            bool isNowGGPlatform = false;
#if !UNITY_EDITOR
            AndroidJavaClass player = new  AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject activity = player.GetStatic<AndroidJavaObject>("currentActivity");

            AndroidJavaClass nowGGClass = new  AndroidJavaClass("gg.now.sdk.NowggDetectionSDK");
            isNowGGPlatform = nowGGClass.CallStatic<bool>("isNowGGPlatform", activity);
#endif
            NowGGLogger.Log($"isNowGGPlatform: {isNowGGPlatform}");
            return isNowGGPlatform;
        }

        public static string GetNowGGDeviceType()
        {
            string deviceType = "NA";
#if !UNITY_EDITOR
            AndroidJavaClass player = new  AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject activity = player.GetStatic<AndroidJavaObject>("currentActivity");

            AndroidJavaClass nowGGClass = new  AndroidJavaClass("gg.now.sdk.NowggDetectionSDK");
            deviceType = nowGGClass.CallStatic<string>("getDeviceTypeNowGG", activity);
#endif
            NowGGLogger.Log($"NowGG device type: {deviceType}");
            return deviceType;
        }

        public static string GetStoreType()
        {
            Debug.Log("GetStoreType() called");
#if !UNITY_EDITOR
            AndroidJavaClass nowGGClass = new AndroidJavaClass("gg.now.sdk.NowggPaymentUnitySDK");
            string storeType = nowGGClass.CallStatic<string>("getStoreType");
            NowGGLogger.Log($"Store type: {storeType}");
            return storeType;
#else
    return "NOT_INIT";
#endif
        }

        public static string GetEnvironmentName()
        {
            Debug.Log("GetEnvironmentName() called");
#if !UNITY_EDITOR
            AndroidJavaClass nowGGClass = new AndroidJavaClass("gg.now.sdk.NowggPaymentUnitySDK");
            string environmentName = nowGGClass.CallStatic<string>("getEnvironmentName");
            NowGGLogger.Log($"Environment name: {environmentName}");
            return environmentName;
#else
    return "NOT_INIT";
#endif
        }

        public void RecordGameEvent(string eventName, string eventData, string userId)
        {
#if !UNITY_EDITOR
            AndroidJavaClass unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject activity = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");

            nowGGAndroidJavaObj.Call("recordGameEvent",activity, eventName, eventData, userId);
#endif
        }
    }
}
